#!/usr/bin/perl
#############################################################################
#
# DLV external builtins test tool
# Author: Susanna Cozza
# Version: 1.0
#
#############################################################################

# module used for options management on command line
use Getopt::Long;
# hash with predefined messages
%BITESTmsg=(
        'usage' =>      <<"EOF",
DLV EXTERNAL BUILTINS TEST TOOL

$ENV{SCRIPT_NAME} builtinName tableName [options]

options:
-s | --source           builtin source file name if different from builtinName,
                        used also as library name if no -l option is specified
        sourceFileName: without .C extension, same path as DLV builtin LIB
-l | --lib              builtin library name if different from builtinName and sourceFileName
        libName:        without .so extension, same path as DLV builtin LIB
-h | --help             show this screen
-e | --extendTable      test on all possible combination of input values in tableName
EOF
        'fileerr'       =>      "Error opening file",
        'err'           =>      "Error running code: "
        );
# end messagesed

# array with all possible options, both in 'long' and 'short' version
# =s stands for an option requiring a string argument
@BITESTcmdline=(
        'source|s=s',
        'lib|l=s',
        'help|h',
        'extendTable|e',
        );
# GetOptions: obtain the options included in the command line, consequently
# setting the BITESTparams hash. If options don't respect the declared syntax
# or the parameters number is different from 2 the usage message is printed
# and the job is ended
if (!GetOptions(\%BITESTparams,@BITESTcmdline) || @ARGV != 2) {
        print STDERR $BITESTmsg{'usage'};
        exit 1;
}
# If 'help' option is set then prints the usage message and the job ends
if ($BITESTparams{'help'}) {
        print $BITESTmsg{'usage'};
        exit 0;
}

$source_file_name = $ARGV[0];
$library_name = $ARGV[0];

# If 'source' option is set then uses it also as library name
if ($BITESTparams{'source'}) {
        $source_file_name = $BITESTparams{'source'};
        $library_name = $BITESTparams{'source'};
} 

# Get environment variables value
$DLV_TO_TEST = $ENV{DLV_TO_TEST};
$PATH_TO_BETL = $ENV{PATH_TO_BETL};
$PATH_TO_LIB = $ENV{PATH_TO_LIB};


# Get from source file the defined oracle patterns
@patterns = ();
$not_defined_patterns = '';
$error_msg = getPatterns($ARGV[0], $source_file_name);
if ($error_msg) {
        print STDERR $error_msg;
        exit 1;
}
if (@patterns == 0) {
        print STDERR "$BITESTmsg{'err'} no valid oracle function defined in source file\n\n";
        exit 1;
}

# Array with temporary file name to be deleted when the job ends
@temp_file_names = ('bitest.betl', 'bitest.betl.betl');

# If 'lib' option is set then uses it else uses the first parameter (builtinName)
if ($BITESTparams{'lib'}) {
        $library_name = $BITESTparams{'lib'};
} 

# Makes the main work to prepare the builtin test
@preds = ();
$error_msg = mainWork($ARGV[0], $ARGV[1], $library_name);
if ($error_msg) {
        print STDERR $error_msg;
        exit 1;
}

# If 'extendTable' option is set then call the 'extendTable' subroutine to find results for
# new combination of input values. The new reults are pushed in the 'ext_preds' array
@ext_preds = ();
if ($BITESTparams{'extendTable'}) {
        $error_msg = extendTable($ARGV[0], $ARGV[1], $library_name);
}
if ($error_msg) {
        print STDERR $error_msg;
        exit 1;
}

# If there is any element in the 'ext_preds' array print it as an extended table values
if (@ext_preds > 0) {
        print "The following extended table values were also generated and exploited for testing:\n\n";
        foreach $ext_pred (@ext_preds) {
                # remove initial predicate name and '('
                $ext_pred =~ s/^.+\(//;
                # remove closing ')'
                $ext_pred =~ s/\)$//;
                print "$ext_pred\n";
        }
        print "\n";
}

# if there are not defined patterns print a message
if ($not_defined_patterns ne '') {
        print "*WARNING* Oracle function(s) for pattern(s): '";
        print "$not_defined_patterns"."' not defined.\n";
}

# Run the BETL compiler on the generated BETL source file
`$PATH_TO_BETL/BETLc bitest.betl`;
# Run the BETL interpreter on the binary generated by the BETL compiler
# -q option stands for quiet =>
#       prevents the interpreter from printing the initial greeting message
# -v lfd option =>
#       print a message only for failed tests, displaying the difference
#       between expected result and test result
@results = `$PATH_TO_BETL/BETLi -q -v lfd bitest.betl.betl`;
# If no report from BETL (all test succeded) then print a success message
if ("@results" eq '') {
        print "OK! All defined oracle functions succeeded.\n";
}
else {
# Filter BETL report giving on the same line both expected and obtained result
        $i=0;
        while ( $i<=$#results) {
                if ($results[$i] =~ /\*FAIL\*/) {
                        print $results[$i];
                        $i++;
                        $test_res = '';
                        # group all report lines for current test
                        while (!($results[$i] =~ /\*FAIL\*/) && $i<=$#results) {
                                $test_res .= $results[$i];
                                $i++;
                        }
                        # all lines about expected results
                        @wants = ($test_res =~ /(=\sline\s+\d+\s+want:\s+.+\\n\s)/g);
                        # all lines about obtained results
                        @gots = ($test_res =~ /(=\sline\s+\d+\s+got:\s+.+\\n\s)/g);
                        $got = shift @gots;
                        foreach $want (@wants) {
                                $want =~ s/\\n\n//;
                                $want =~ s/want:/wanted:/;
                                # need to make a copy becouse later we use the line number info
                                my $toprint = $want;
                                $toprint =~ s/\sline\s+\d+/===>/;
                                print $toprint;
                                # if there is a corresponding line with obtained result
                                if (substr($want,0,11) eq substr($got,0,11)) {
                                        $got =~ s/\\n//;
                                        # print the last part of it
                                        print substr($got,12);
                                        $got = shift @gots;
                                } else {
                                        # no result obtained for that input value
                                        print " got: failed\n";
                                }
                        }
                } else {
                        print $results[$i];
                        $i++;
                }
        }
}

exit 0;

#############################################################################
# getPatterns subroutine
#
# Get from source file defined oracle patterns. Generate all possible patterns
# except that including letter 'o' only. Print a warning message showing the
# list of not defined patterns.
#
#############################################################################
sub getPatterns {
        # get subroutine parameter and open source file
        my $builtin_name = @_[0];
        my $source_file_name = @_[1];
        open(FILES, "< $PATH_TO_LIB$source_file_name.C") ||
                return "$BITESTmsg{'fileerr'} $PATH_TO_LIB$source_file_name.C\n\n";
        # read it
        my @sourcefile = <FILES>;
        # and close
        close(FILES);

        # all source file content in a string variable
        my $sourcecode = "@sourcefile";
        # put in the 'patterns' array all patterns (sequence of letters 'i'
        # and 'o') defined in a row containing: the 'BUILTIN' word, any
        # whitespace character, a '(', any whitespace character, the name of the 
        # builtin,  any whitespace character, any sequence of letters 'i'
        # and 'o' (this is the matching to be extracted), any whitespace
        # character, a ')'. The /g modifier is used to allow a global matching
        # that is matching within a string as many times as possible.
        @patterns = ($sourcecode =~ /BUILTIN\s*\(\s*$builtin_name\s*,\s*([io]+)\s*\)/g);

        # number of elements in a pattern
        my $param_num = split(//,$patterns[0]);
        # all possible patterns having 1 element
        my @base_patterns = ('i','o');
        my @all_patterns;
        for (my $j=1; $j<$param_num; $j++) {
                @all_patterns = ();
                # each base pattern is used to generate a pattern having one
                # more element it is obtained concatenating a letter 'i' and
                # then a letter 'o' to the base pattern
                foreach my $pattern (@base_patterns) {
                        push(@all_patterns,"$pattern".'i');
                        push(@all_patterns,"$pattern".'o');
                }
                @base_patterns = @all_patterns;
        }
        # the last pattern contains only letter 'o' so is popped
        pop(@all_patterns);
        # hash with the defined patterns
        my %def_patterns = ();
        foreach my $pattern (@patterns) {
                $def_patterns{$pattern} = 1;
        }
        foreach my $a_pattern(@all_patterns) {
                # if a pattern is not in the def_patterns hash it is added to
                # the not_defined_patterns string
                if (!$def_patterns{$a_pattern}) {
                        $not_defined_patterns .= "$a_pattern ";
                }
        }
        # remove the last whitespace
        $not_defined_patterns =~ s/ $//;
        return '';
}

#############################################################################
# mainWork subroutine
#
# Generate dlv testing programs for all defined patterns.
# Define BETL source file to test generated programs.
# Create a file with the test expected results.
#
#############################################################################
sub mainWork {
        # get subroutine parameters
        my $builtin_name = @_[0];
        my $table_name = @_[1];
        my $library_name = @_[2];

        # open value's table file
        open(FILEV, "< $table_name") ||
                return "$BITESTmsg{'fileerr'} $table_name\n\n";
        # read it
        my @valuefile = <FILEV>;
        # remove trailing newlines character
        chomp @valuefile;
        # close it
        close(FILEV);

        # open file with fixed stuff for the BETL source
        open(FILETP, "< bitestprefix.betl") ||
                return "$BITESTmsg{'fileerr'} bitestprefix.betl\n\n";
        # read it
        #
        my @testp = <FILETP>;
        # close it
        close(FILETP);
        # create the file used as BETL source
        open(FILET, "> bitest.betl") ||
                return "$BITESTmsg{'fileerr'} bitest.betl\n\n";
        # print the fixed stuff
        print FILET @testp;

        # for each defined pattern ...
        for (my $j=0; $j<=$#patterns; $j++) {
                my $pattern = $patterns[$j];
                # create the dlv program file
                open(FILEP, "> $builtin_name\_$pattern.dl") ||
                        return "$BITESTmsg{'fileerr'} $builtin_name\_$pattern.dl\n\n";
                # it is a temporary file to be deleted at the end
                push(@temp_file_names, "$builtin_name\_$pattern.dl");
                print FILEP "#include <$library_name>\n\n";

                my $maxint = 0;
                # for each row in the value's table
                foreach my $row (@valuefile) {
                        # do not consider row with only whitespace characters
                        next if ($row =~ /^\s*$/);

                        my @pattern_el = split(//, $pattern);
                        # remove all whitespace characters
                        $row =~ s/\s//g;
                        # take single values separated by ',' and put in an array
                        use Text::ParseWords;
                        my @values = parse_line(",", 1, $row);
                        # return an error message if pattern elements number
                        # is different from values number
                        if ($#pattern_el != $#values) {
                                return "$BITESTmsg{'err'} wrong arity in table's row '$row'\n\n";
                        }
                        # just for the first pattern store row values in the 'preds'
                        # array, used to build the expected result file
                        if ($j == 0) {
                                push(@preds,"$builtin_name($row)");
                        }

                        my $num_out = 1;
                        # for each value ...
                        for (my $i=0; $i<=$#values; $i++) {
                                # if it is a number and is greater than maxint,
                                # this is the new maxint value
                                if ($values[$i] =~ /\d+/ && $values[$i] > $maxint) {
                                        $maxint = $values[$i];
                                }
                                # if pattern element in the same position is 'o'
                                # then substitute the value with a variable named 'OUTPUTx'
                                # where x is the i-th output variable
                                if ($pattern_el[$i] eq 'o') {
                                        $values[$i] = "OUTPUT$num_out";
                                        $num_out++;
                                }
                                # pattern element must be either 'o' or 'i'
                                elsif ($pattern_el[$i] ne 'i') {
                                        return "$BITESTmsg{'err'} wrong pattern '$pattern'\n\n";
                                }
                        }
                        my $pred_arg = join(',', @values);
                        # print on the dlv program file the rule
                        print FILEP "$builtin_name($pred_arg) :- #$builtin_name($pred_arg).\n";
                }
                # add maxint directive if necessary
                if ($maxint > 0) {
                        print FILEP "#maxint = $maxint.\n";
                }
                # close dlv program file
                close(FILEP);

                # add to BETL source file the TEST section
                print FILET "\t<TEST>\n\t\t<NAME>$builtin_name\_$pattern</NAME>\n";
                print FILET "\t\t<ARGS>$builtin_name\_$pattern.dl</ARGS>\n";
                print FILET "\t\t<OUT><FILE>result/$builtin_name</FILE></OUT>\n\t</TEST>\n";

        }

        # complete BETL source file
        print FILET "</BODY>\n\n</BETL>\n";
        # close it
        close(FILET);

        mkdir result;
        # create the expected result file
        open(FILEM, "> result/$builtin_name") ||
                return "$BITESTmsg{'fileerr'} result/$builtin_name\n\n";
        # it is a temporary file to be deleted at the end
        push(@temp_file_names, "result/$builtin_name");
        # expected result predicates are sorted and joined using '\n' as separator
        my $modello = join("\n",sort(@preds));
        # and printed on the result file
        print FILEM "$modello";
        # close result file
        close(FILEM);
        return '';
}

#############################################################################
# extendTable subroutine
#
# Generate dlv testing programs combining input values to generate extended
# table values. Run dlv on this programs and store in the 'ext_preds' array
# resulting predicates not included in the already obtained results.
#
#############################################################################
sub extendTable {
        # get subroutine parameters
        my $builtin_name = @_[0];
        my $table_name = @_[1];
        my $library_name = @_[2];

        # open value's table file
        open(FILEV, "< $table_name") ||
                return "$BITESTmsg{'fileerr'} $table_name\n\n";
        # read it
        my @valuefile = <FILEV>;
        # remove trailing newlines character
        chomp @valuefile;
        # close it
        close(FILEV);

        # hash with already obtained resulting predicates
        my %preds_result = ();
        foreach my $pred(@preds) {
                $preds_result{$pred} = 1;
        }

        # for each defined pattern ...
        foreach my $pattern(@patterns) {
                # create the extended table dlv program file
                open(FILEP, "> ex\_$builtin_name\_$pattern.dl") ||
                        return "$BITESTmsg{'fileerr'} ex\_$builtin_name\_$pattern.dl\n\n";
                # it is a temporary file to be deleted at the end
                push(@temp_file_names, "ex\_$builtin_name\_$pattern.dl");
                print FILEP "#include <$library_name>\n\n";

                my $maxint = 0;
                # array with value's types
                my @types = ();
                
                # for each row in the value's table
                foreach my $row (@valuefile) {
                        # do not consider row with only whitespace characters
                        next if ($row =~ /^\s*$/);
                        # remove all whitespace characters
                        $row =~ s/\s//g;
                        # take single values separated by ',' and put in an array
                        use Text::ParseWords;
                        my @values = parse_line(",", 1, $row);
                        # index for value position
                        my $vindex = 0;

                        # for each value ...
                        foreach my $value (@values) {
                                if ($value =~ /^".*"$/) {
                                        # current value is a string so print it
                                        # in the dlv program file as a fact for
                                        # the 'string' predicate
                                        print FILEP "string($value).\n";
                                        # set the type for this value
                                        $types[$vindex] = 'string';
                                } else {
                                        if ($value =~ /\d+/) {
                                                # current value is a number so print
                                                # it in the dlv program file
                                                # as a fact for the 'number' predicate
                                                print FILEP "number($value).\n";
                                                # set the type for this value
                                                $types[$vindex] = 'number';
                                                # if it is greater than maxint, this
                                                #is the new maxint value
                                                if ($value > $maxint) {
                                                        $maxint = $value;
                                                }
                                        } else {
                                                # current value is a symbol so print
                                                # it in the dlv program file
                                                # as a fact for the 'symbol' predicate
                                                print FILEP "symbol($value).\n";
                                                # set the type for this value
                                                $types[$vindex] = 'symbol';
                                        }
                                }
                                $vindex++;
                        }
                }

                my $num_in = 0;
                my $num_out = 0;
                my $input_pred = "";
                my @pattern_els = split(//, $pattern);
                my @pred_args = ();
                # for each pattern element ...
                foreach my $pattern_el(@pattern_els) {
                        # if it is 'o' then put in the 'pred_args' array 'OUTPUTx'
                        # where x is the i-th output variable
                        if ($pattern_el eq 'o') {
                                push(@pred_args, "OUTPUT$num_out");
                                $num_out++;
                        }
                        # if it is 'i' then put in the 'pred_args' array 'INTPUTx'
                        # where x is the i-th input variable
                        # and add to the 'input_pred' string the term 'input(INPUTx), '
                        else {
                                push(@pred_args, "INPUT$num_in");
                                $input_pred .= "$types[$num_in+$num_out](INPUT$num_in), ";
                                $num_in++;
                        }
                }
                my $pred_arg = join(',', @pred_args);
                # print on the extended table dlv program file the rule
                print FILEP "\n$builtin_name($pred_arg) :- $input_pred\#$builtin_name($pred_arg).\n";

                # add maxint directive if necessary
                if ($maxint > 0) {
                        print FILEP "\n#maxint = $maxint.\n";
                }
                # close extended table dlv program file
                close(FILEP);
                # run dlv on the generated program file
                my $result = `$DLV_TO_TEST -silent -nofacts -libpath=$ENV{PATH_TO_LIB} ex\_$builtin_name\_$pattern.dl`;
                # if there is a resulting non empty model
                if ($result =~ /^{(.+)}$/) {
                        # take single resulting predicate separated by ','
                        # and put in an array
                        my @ext_ress = split(/, /, $1);
                        # for each resulting predicate ...
                        foreach my $ext_res(@ext_ress) {
                                # if it is not present in the already obtained results,
                                # put it in the 'ext_preds' array
                                if (!$preds_result{$ext_res}) {
                                        push(@ext_preds, $ext_res);
                                        $preds_result{$ext_res} = 1;
                                }
                        }
                }
        }

        return '';
}

#############################################################################
# END subroutine
#
# Function triggered when leaving the script, it is used to clean-up reasons
#
#############################################################################
END {
        # remove all temporary files
        unlink(@temp_file_names);
        # remove the temporary 'result' directory
        rmdir(result);
}
