package saw.reflection.show.classSpy;
import static java.lang.System.out;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.ArrayList;
import java.util.List;

/**
 * @author gio
 * 
 */
public class ClassSpy {

	private final Class<?> clazz;

	public static BeanInfo getBeanInfo(Class<?> clazz) {
		try {
			return Introspector.getBeanInfo(clazz);
		} catch (final IntrospectionException e) {
			throw new RuntimeException("Can't get the BeanInfo", e);
		}
	}

	/**
	 * 
	 * @param clazz
	 */
	public ClassSpy(Class<?> clazz) {
		this.clazz = clazz;
	}

	/**
	 * @param clazz
	 */
	public void print() {
		classDeclarationSpy();
		classDetailSpy();
	}

	/**
	 * Prints recursively the ancestors of a given class
	 * 
	 * @param clazz
	 *            the class
	 * @param l
	 *            the list of ancestors
	 */
	private void printAncestor(Class<?> clazz, List<Class> l) {
		Class<?> ancestor = clazz.getSuperclass();
		if (ancestor != null) {
			l.add(ancestor);
			printAncestor(ancestor, l);
		}
	}

	/**
	 * Prints info about a class declaration
	 * 
	 * @param clazz
	 *            the class to spy
	 */
	private void classDeclarationSpy() {
		out.format("Class:%n  %s%n%n", clazz.getCanonicalName());
		out.format("Modifiers:%n  %s%n%n", Modifier.toString(clazz.getModifiers()));

		out.format("Type Parameters:%n");
		TypeVariable[] tv = clazz.getTypeParameters();
		if (tv.length != 0) {
			out.format("  ");
			for (TypeVariable t : tv)
				out.format("%s ", t.getName());
			out.format("%n%n");
		} else {
			out.format("  -- No Type Parameters --%n%n");
		}

		out.format("Implemented Interfaces:%n");
		Type[] intfs = clazz.getGenericInterfaces();
		if (intfs.length != 0) {
			for (Type intf : intfs)
				out.format("  %s%n", intf.toString());
			out.format("%n");
		} else {
			out.format("  -- No Implemented Interfaces --%n%n");
		}

		out.format("Inheritance Path:%n");
		List<Class> l = new ArrayList<Class>();
		printAncestor(clazz, l);
		if (l.size() != 0) {
			for (Class<?> cl : l)
				out.format("  %s%n", cl.getCanonicalName());
			out.format("%n");
		} else {
			out.format("  -- No Super Classes --%n%n");
		}

		out.format("Annotations:%n");
		Annotation[] ann = clazz.getAnnotations();
		if (ann.length != 0) {
			for (Annotation a : ann)
				out.format("  %s%n", a.toString());
			out.format("%n");
		} else {
			out.format("  -- No Annotations --%n%n");
		}
	}

	/**
	 * Prints info about packages, constructors, fields and methods about a
	 * given class
	 * 
	 * @param clazz
	 */
	private void classDetailSpy() {

		Package p = clazz.getPackage();
		out.format("Package:%n  %s%n%n", (p != null ? p.getName()
				: "-- No Package --"));

		printMembers(clazz.getConstructors(), "Constuctors");
		printMembers(clazz.getFields(), "Fields");
		printMembers(clazz.getDeclaredMethods(), "Declared Methods");
		printMembers(clazz.getMethods(), "All Methods");

	}

	/**
	 * 
	 * @param mbrs
	 * @param s
	 */
	private void printMembers(Member[] mbrs, String s) {
		out.format("%s:%n", s);
		for (Member mbr : mbrs) {
			if (mbr instanceof Field)
				out.format("  %s%n", ((Field) mbr).toGenericString());
			else if (mbr instanceof Constructor)
				out.format("  %s%n", ((Constructor) mbr).toGenericString());
			else if (mbr instanceof Method)
				out.format("  %s%n", ((Method) mbr).toGenericString());
		}
		if (mbrs.length == 0)
			out.format("  -- No %s --%n", s);
		out.format("%n");
	}

}
